/*==========================================================================
* HWCURSOR.C - Copyright (c) 1994 ATI Technologies Inc. All rights reserved*
*                                                                          *
* PGL functions to define, enable, disable, and position the Mach64        *
* hardware cursor. This includes support for the TVP3026 and IBM514 DACs.  *
* The routines are structured so that the DAC type is transparent to the   *
* primary hardware cursor functions.                                       *
* ======================================================================== */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <conio.h>

#include "..\inc\atim64.h"
#include "..\inc\pgl.h"
#include "..\inc\pglglob.h"

// Private copy of the cursor data
PGL_hwcursor    PGL_cursor;

/* --------------------------------------------------------------------------
  pgl_setcursorcolors - set hardware cursor colors

  This function is called exclusively by PGL_sethwcursor().
-------------------------------------------------------------------------- */
void pgl_setcursorcolors(unsigned long color0, unsigned long color1)
{
    unsigned long curclr0, curclr1;
    unsigned long red0, green0, blue0, red1, green1, blue1;
    unsigned long rshift, gshift, bshift;
    unsigned long temp;
    int index;
    PGL_palette entry;

    // setup cursor colors for 4, 8 bpp modes
    if ((PGL_modecfg.bpp == 4) || (PGL_modecfg.bpp == 8))
    {
        // get color componets from palette using color indices
        if ((PGL_modecfg.dac_attribute == DAC_SP_ATI68860) ||
            (PGL_modecfg.dac_attribute == DAC_SP_TVP3026) ||
            (PGL_modecfg.dac_attribute == DAC_SP_IBM514) ||
            (PGL_modecfg.chip_type == CHIP_CT_ID) ||
            (PGL_modecfg.chip_type == CHIP_DT_ID))
        {
            // ensure palette registers are available
            iow8(ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xfc);

            // Cursor colors are 8 bit values, palette colors are 6 bit values
            entry = PGL_getpalette((int)(color0 & 0xff));
            red0 = (unsigned long)(entry.red) << 2;
            green0 = (unsigned long)(entry.green) << 2;
            blue0 = (unsigned long)(entry.blue) << 2;
            entry = PGL_getpalette((int)(color1 & 0xff));
            red1 = (unsigned long)(entry.red) << 2;
            green1 = (unsigned long)(entry.green) << 2;
            blue1 = (unsigned long)(entry.blue) << 2;

            // assign cursor color values for Mach64 Bedrock internal DACs
            if ((PGL_modecfg.chip_type == CHIP_CT_ID) ||
                (PGL_modecfg.chip_type == CHIP_DT_ID))
            {
                curclr0 = (red0 << 24) | (green0 << 16) | (blue0 << 8) | color0;
                curclr1 = (red1 << 24) | (green1 << 16) | (blue1 << 8) | color1;
            }
        }
        else // use color indices for standard DACs
        {
            curclr0 = (int)(color0 & 0xff);
            curclr1 = (int)(color1 & 0xff);
        }
    }
    else // setup cursor color for 15, 16, 24, 32 bpp modes
    {
        // determine color component shifting values
        switch(PGL_modecfg.bpp)
        {
            case 16:
                if (PGL_modecfg.depth == 555)
                {
                    rshift = 3;
                    gshift = 3;
                    bshift = 3;
                }
                else  // 565 weight
                {
                    rshift = 3;
                    gshift = 2;
                    bshift = 3;
                }
                break;

            case 24:
            case 32:
                rshift = 0;
                gshift = 0;
                bshift = 0;
                break;
        }

        // get cursor color 0 components
        red0 = PGL_getprimarycolor(RED, color0) << rshift;
        green0 = PGL_getprimarycolor(GREEN, color0) << gshift;
        blue0 = PGL_getprimarycolor(BLUE, color0) << bshift;

        // get cursor color 1 components
        red1 = PGL_getprimarycolor(RED, color1) << rshift;
        green1 = PGL_getprimarycolor(GREEN, color1) << gshift;
        blue1 = PGL_getprimarycolor(BLUE, color1) << bshift;

        // special case for Brooktree DAC in 24 (BGR) bpp mode
        if ((PGL_modecfg.bpp == 24) &&
            (PGL_modecfg.depth == DEPTH_24_BGR) &&
            (PGL_querydata.dac_type == DAC_BT481))
        {
            // swap RED and BLUE components
            temp = blue0;
            blue0 = red0;
            red0 = temp;
            temp = blue1;
            blue1 = red1;
            red1 = temp;
        }

        // assign cursor color values for standard DACs
        if ((PGL_modecfg.dac_attribute != DAC_SP_ATI68860) &&
            (PGL_modecfg.dac_attribute != DAC_SP_TVP3026) &&
            (PGL_modecfg.dac_attribute != DAC_SP_IBM514))
        {
            curclr0 = (red0 << 24) | (green0 << 16) | (blue0 << 8);
            curclr1 = (red1 << 24) | (green1 << 16) | (blue1 << 8);
        }
    }

    // write to hardware to set cursor colors for ATI68860,TVP3026,IBM514 DACs
    if ((PGL_modecfg.dac_attribute == DAC_SP_ATI68860) ||
        (PGL_modecfg.dac_attribute == DAC_SP_TVP3026) ||
        (PGL_modecfg.dac_attribute == DAC_SP_IBM514))
    {
        // special setup for ATI68860, TVP3026, and IBM514 DACs for cursor
        // colors

        // access cursor color registers
        iow8(ioDAC_CNTL, (ior8(ioDAC_CNTL) & 0xfc) | 1);

        // select 1st cursor color register set
        switch(PGL_modecfg.dac_attribute)
        {
            case DAC_SP_ATI68860:
                iow8(ioDAC_REGS + DAC_W_INDEX, 0);
                index = DAC_DATA;
                break;

            case DAC_SP_TVP3026:
                // skip OVERSCAN color register on TVP3026 DAC
                iow8(ioDAC_REGS + DAC_W_INDEX, 1);
                index = DAC_DATA;
                break;

            case DAC_SP_IBM514:
                iow8(ioDAC_REGS + DAC_R_INDEX, 1);      // auto-increment
                iow8(ioDAC_REGS + DAC_W_INDEX, 0x40);
                iow8(ioDAC_REGS + DAC_DATA, 0);         // index 0x40
                index = DAC_MASK;
                break;
        }

        // set 1st cursor color - 8 bit values required
        iow8(ioDAC_REGS + index, (int)red0);
        iow8(ioDAC_REGS + index, (int)green0);
        iow8(ioDAC_REGS + index, (int)blue0);

        // set 2nd cursor color - 8 bit values required
        iow8(ioDAC_REGS + index, (int)red1);
        iow8(ioDAC_REGS + index, (int)green1);
        iow8(ioDAC_REGS + index, (int)blue1);

        // return DAC register set to palette registers
        iow8(ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xfc);
    }
    else // write to hardware to set cursor colors for standard DACs
    {
        regw(CUR_CLR0, curclr0);
        regw(CUR_CLR1, curclr1);
    }
}

/* --------------------------------------------------------------------------
  PGL_sethwcursor - define hardware cursor bitmap, colors, and hot spot

  This function sets up the hardware cursor data region according to the
  given bitmap data. The data region is located at (0, y). It is recommended
  that 'y' be set to (maximum y - 64 lines). This will insure that the
  data will not be corrupted by other PGL functions such as polygon draws.
  If the DAC type is a TVP3026 or IBM514, the hardware cursor bitmap is
  uploaded to the DAC itself and not to video memory (this DAC supports its
  own hardware cursor). This routine will convert the data and positioning as
  necessary so that the interface remains the same. The hardware cursor may
  vary in size from 1x1 to 64x64. The hot spot serves as the reference point
  for positioning the hardware cursor and this reference must be within the
  visible hardware cursor area. The hot spot uses the left-top corner of the
  visible region of the hardware cursor as its reference. Note that the hot
  spot is a zero-based quantity. The expected bitmap format is interpreted
  from LSB to MSB for each data word. The LSB will be drawn first in a left
  to right direction. Since the cursor position is NOT set in this routine,
  the position should be set by calling PGL_sethwcursorpos() prior to
  enabling the cursor. If any of the input parameters are invalid, YES_ERROR
  is returned. Normally, NO_ERROR is returned.
-------------------------------------------------------------------------- */
int PGL_sethwcursor(int y, int width, int height, int hot_x, int hot_y,
                    unsigned long color0, unsigned long color1,
                    unsigned int *bitmap)
{
    unsigned long cur_offset, cur_size;
    unsigned long temp1, temp2, temp3, temp4, temp5, temp6;
    unsigned long bitdata;
    int i, index, dataindex, start, widthwords;
    unsigned int cur_data, temp;

    // Check that cursor size is within limits
    if ((width < 1) || (width > 64)) return (YES_ERROR);
    if ((height < 1) || (height > 64)) return (YES_ERROR);
    if ((hot_x < 0) || (hot_x >= width)) return (YES_ERROR);
    if ((hot_y < 0) || (hot_y >= height)) return (YES_ERROR);

    // record cursor definition to local static buffer
    PGL_cursor.y = y;
    PGL_cursor.width = width;
    PGL_cursor.height = height;
    PGL_cursor.hot_x = hot_x;
    PGL_cursor.hot_y = hot_y;
    PGL_cursor.color0 = color0;
    PGL_cursor.color1 = color1;

    // set hwcursor bitmap to transparent
    for (index = 0; index < (HWCURHEIGHT * HWCURWIDTH); index++)
	{
        PGL_cursor.bitmap[index] = 0xaaaa;
	}

    //
    // Overlay the user hwcursor data onto the transparent bitmap. The data
    // is loaded into the top-right corner of the 64x64 bitmap.
    //
    dataindex = 0;
    widthwords = width / 8;
    if (width > widthwords * 8)
    {
        widthwords++;
    }
    start = HWCURWIDTH - widthwords;
    for (index = start; index < (HWCURWIDTH * height); index = index + HWCURWIDTH)
	{
        i = 0;
        do
        {
            PGL_cursor.bitmap[index + i] = *(bitmap + dataindex);
            if (width < 8)
            {
                PGL_cursor.bitmap[index + i] = PGL_cursor.bitmap[index + i] << ((8 - width) * 2);
            }
            dataindex++;
            i++;
        } while (i < widthwords);
    }

    // calculate offset in qwords for location of cursor data in video memory
    cur_offset = pgl_getxyoffset(0, y) / 8;
    PGL_cursor.cur_offset = cur_offset;

    //
    // Convert data and write to DAC hardware cursor if TVP3026 DAC. The even
    // bits are stripped from the bitmap filled above and are written to
    // plane 0 of the TVP3026 DAC (first 512 bytes). Plane 1 (second 512
    // bytes) is written using the odd bits. Also, the bit positions are
    // reversed. Note that the visible cursor data will be placed in the
    // top-right corner of the 64x64 cursor region.
    //
    // Also if the DAC is an IBM514, it requires special programming in the
    // same manner as the TVP3026.
    //
    if (PGL_modecfg.dac_attribute == DAC_SP_TVP3026)
    {
        // set cursor RAM write address to 0
        iow8(ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xfc);
        iow8(ioDAC_REGS + DAC_W_INDEX, 0);

        // select cursor RAM data register - auto increments with each write
        iow8(ioDAC_CNTL, (ior8(ioDAC_CNTL) & 0xfc) | 2);

        // plane select loop
        for (i = 0; i < 2; i++)
        {
            // convert data and write cursor data (2 planes, 512 bytes/plane)
            for (index = 0; index < 512; index++)
            {
                // 2 bytes written per cursor data word
                temp = PGL_cursor.bitmap[index];
                if (i == 0)
                {
                    // plane 0 byte data
                    cur_data = (((temp & 0x4000) >> 14) | ((temp & 0x1000) >> 11) |
                                ((temp & 0x0400) >> 8) | ((temp & 0x0100) >> 5) |
                                ((temp & 0x0040) >> 2) | ((temp & 0x0010) << 1) |
                                ((temp & 0x0004) << 4) | ((temp & 0x0001) << 7));
                }
                else
                {
                    // plane 1 byte data
                    cur_data = (((temp & 0x8000) >> 15) | ((temp & 0x2000) >> 12) |
                                ((temp & 0x0800) >> 9) | ((temp & 0x0200) >> 6) |
                                ((temp & 0x0080) >> 3) | (temp & 0x0020) |
                                ((temp & 0x0008) << 3) | ((temp & 0x0002) << 6));
                }
                iow8(ioDAC_REGS + DAC_R_INDEX, (int)(cur_data & 0xff));
            }
        }

        // select default palette registers
        iow8(ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xfc);
    }
    else if (PGL_modecfg.dac_attribute == DAC_SP_IBM514)
    {
        // select index registers
        iow8(ioDAC_CNTL, (ior8(ioDAC_CNTL) & 0xfc) | 1);

        // enable auto-increment
        iow8(ioDAC_REGS + DAC_R_INDEX, 1);

        // set cursor RAM write address to start of cursor array - offset 100h
        iow8(ioDAC_REGS + DAC_W_INDEX, 0);
        iow8(ioDAC_REGS + DAC_DATA, 1);

        // write given cursor data to cursor array on chip - same as Mach64
        // hardware cursor interface
        for (index = 0; index < 512; index++)
        {
            // 2 bytes per loop
            temp = PGL_cursor.bitmap[index];
            iow8(ioDAC_REGS + DAC_MASK, (int)(temp & 0xff));
            iow8(ioDAC_REGS + DAC_MASK, (int)((temp >> 8) & 0xff));
        }

        // set hot-spot registers
        iow8(ioDAC_REGS + DAC_W_INDEX, 0x35);
        iow8(ioDAC_REGS + DAC_DATA, 0);
        iow8(ioDAC_REGS + DAC_MASK, 64 - width + hot_x);
        iow8(ioDAC_REGS + DAC_MASK, hot_y);

        // select default palette registers
        iow8(ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xfc);
    }
    else // write to video memory for other DACs (Mach64 hardware cursor)
    {
        // Use 16 bpp to upload hardware cursor bitmap to video memory

        // save vital registers
        PGL_waitforidle();
        temp1 = regr(DP_PIX_WIDTH);
        temp2 = regr(DP_CHAIN_MASK);
        temp3 = regr(DST_OFF_PITCH);
        temp4 = regr(DP_SRC);
        temp5 = regr(DP_MIX);
        temp6 = regr(DST_CNTL);

        // load bitmap data to hardware cursor bitmap data area
        regw(DP_PIX_WIDTH, (temp1 & BYTE_ORDER_LSB_TO_MSB) |
                           HOST_16BPP | SRC_16BPP | DST_16BPP);
        regw(DP_CHAIN_MASK, 0x8410);
        regw(DST_OFF_PITCH, 0x00400000 | cur_offset); // pitch = 8
        regw(DP_SRC, FRGD_SRC_HOST);
        regw(DP_MIX, FRGD_MIX_S | BKGD_MIX_D);
        regw(DST_CNTL, DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT);
        regw(DST_Y_X, 0);
        regw(DST_HEIGHT_WIDTH, ((unsigned long)(HWCURWIDTH) << 16) | HWCURHEIGHT);
        for (index = 0; index < (HWCURHEIGHT * HWCURWIDTH * 2); index = index + 2)
        {
            PGL_waitforfifo(1);
            bitdata = (unsigned long)(PGL_cursor.bitmap[index+1]);
            bitdata = (bitdata << 16) | (PGL_cursor.bitmap[index]);
            regw(HOST_DATA0, bitdata);
        }
        PGL_waitforidle();

        // restore vital registers
        regw(DP_PIX_WIDTH, temp1);
        regw(DP_CHAIN_MASK, temp2);
        regw(DST_OFF_PITCH, temp3);
        regw(DP_SRC, temp4);
        regw(DP_MIX, temp5);
        regw(DST_CNTL, temp6);

        // set cursor size offsets
        cur_size = (unsigned long)(64 - height);
        cur_size = (unsigned long)((cur_size << 16) | (64 - width));
        regw(CUR_HORZ_VERT_OFF, cur_size);

        // set offset to cursor data region
        regw(CUR_OFFSET, cur_offset);
    }

    // set hardware cursor colors 0 & 1
    pgl_setcursorcolors(color0, color1);

    return (NO_ERROR);
}

/* --------------------------------------------------------------------------
  PGL_enablehwcursor - turn on the hardware cursor
-------------------------------------------------------------------------- */
void PGL_enablehwcursor(void)
{
    // enable hardware cursor
    regw(GEN_TEST_CNTL, regr(GEN_TEST_CNTL) | HWCURSOR_ENABLE);

    // special enabling required for TVP3026 DAC
    if (PGL_modecfg.dac_attribute == DAC_SP_TVP3026)
    {
        // access cursor control register
        iow8(ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xfc);
        iow8(ioDAC_REGS + DAC_W_INDEX, 6);                 // register 6
        iow8(ioDAC_CNTL, (ior8(ioDAC_CNTL) & 0xfc) | 2);
        iow8(ioDAC_REGS + DAC_MASK, 2);                    // XGA cursor type
        iow8(ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xfc);
    }

    // special enabling required for IBM514 DAC
    if (PGL_modecfg.dac_attribute == DAC_SP_IBM514)
    {
        // access cursor control register
        iow8(ioDAC_CNTL, (ior8(ioDAC_CNTL) & 0xfc) | 1);
        iow8(ioDAC_REGS + DAC_R_INDEX, 1);                 // auto-increment
        iow8(ioDAC_REGS + DAC_W_INDEX, 0x30);
        iow8(ioDAC_REGS + DAC_DATA, 0);                    // index 0x0030
        iow8(ioDAC_REGS + DAC_MASK, 0x0e);                 // XGA cursor type
        iow8(ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xfc);
    }
}

/* --------------------------------------------------------------------------
  PGL_disablehwcursor - turn off the hardware cursor
-------------------------------------------------------------------------- */
void PGL_disablehwcursor(void)
{
    // disable hardware cursor
    regw(GEN_TEST_CNTL, regr(GEN_TEST_CNTL) & (~HWCURSOR_ENABLE));

    // special disabling required for TVP3026 DAC
    if (PGL_modecfg.dac_attribute == DAC_SP_TVP3026)
    {
        // access cursor control register
        iow8(ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xfc);
        iow8(ioDAC_REGS + DAC_W_INDEX, 6);                 // register 6
        iow8(ioDAC_CNTL, (ior8(ioDAC_CNTL) & 0xfc) | 2);
        iow8(ioDAC_REGS + DAC_MASK, 0);                    // disable
        iow8(ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xfc);
    }

    // special enabling required for IBM514 DAC
    if (PGL_modecfg.dac_attribute == DAC_SP_IBM514)
    {
        // access cursor control register
        iow8(ioDAC_CNTL, (ior8(ioDAC_CNTL) & 0xfc) | 1);
        iow8(ioDAC_REGS + DAC_R_INDEX, 1);                 // auto-increment
        iow8(ioDAC_REGS + DAC_W_INDEX, 0x30);
        iow8(ioDAC_REGS + DAC_DATA, 0);                    // index 0x0030
        iow8(ioDAC_REGS + DAC_MASK, 0);                    // disable
        iow8(ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xfc);
    }
}

/* --------------------------------------------------------------------------
  PGL_sethwcursorpos - set the hardware cursor position

  The cursor data offset, vertical offset, horizontal offset, and position
  must be adjusted so that the hot spot can be positioned from (0, 0) to
  (xres-1, yres-1). The Mach64 hardware cursor requires some special
  programming to properly hide the left-top corner of the hot spot when the x
  or y coordinate is less than their respective hot spot offset (the hot spot
  uses the top-left corner of the visible cursor area as its reference).
  If the DAC is a TVP3026, the hardware cursor on the DAC is used. Its
  reference is located at the bottom-right corner of the 64x64 cursor area.
-------------------------------------------------------------------------- */
void PGL_sethwcursorpos(int x, int y)
{
    unsigned long cur_pos, cur_size, cur_offset;

    // check for coordinate violations
    if (x < 0) x = 0;
    if (y < 0) y = 0;
    PGL_cursor.current_x = x;
    PGL_cursor.current_y = y;

    // set cursor position
    if (PGL_modecfg.dac_attribute == DAC_SP_TVP3026)
    {
        // TVP3026 DAC
        cur_pos = (unsigned long)(y + 64 - PGL_cursor.hot_y);
        cur_pos = (unsigned long)((cur_pos << 16) | (x + PGL_cursor.width - PGL_cursor.hot_x));
        regw(DAC_CNTL, (regr(DAC_CNTL) & 0xfffffffc) | 3);
        regw(DAC_REGS + DAC_W_INDEX, cur_pos);
        regw(DAC_CNTL, regr(DAC_CNTL) & 0xfffffffc);
    }
    else if (PGL_modecfg.dac_attribute == DAC_SP_IBM514)
    {
        // IBM514 DAC
        cur_pos = (unsigned long)(y);
        cur_pos = (unsigned long)((cur_pos << 16) | x);
        iow8(ioDAC_CNTL, (ior8(ioDAC_CNTL) & 0xfc) | 1);
        iow8(ioDAC_REGS + DAC_R_INDEX, 1);                 // auto-increment
        iow8(ioDAC_REGS + DAC_W_INDEX, 0x31);
        iow8(ioDAC_REGS + DAC_DATA, 0);                    // index = 0x31
        iow8(ioDAC_REGS + DAC_MASK, (int)(cur_pos & 0xff));         // X low
        iow8(ioDAC_REGS + DAC_MASK, (int)((cur_pos >> 8) & 0xff));  // X high
        iow8(ioDAC_REGS + DAC_MASK, (int)((cur_pos >> 16) & 0xff)); // Y low
        iow8(ioDAC_REGS + DAC_MASK, (int)((cur_pos >> 24) & 0xff)); // Y high
        iow8(ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xfc);
    }
    else // standard DACs
    {
        // set cursor position so that (x, y) equals the hot spot
        if (y < PGL_cursor.hot_y)
        {
            //
            // Adjust CUR_OFFSET to point (hot_y - y) lines into the cursor
            // data region. Each line is 2 bytes wide. Also increase
            // CUR_VERT_OFF by the same number of lines that CUR_OFFSET was
            // adjusted by. Since the above registers are being used to
            // adjust the cursor position, set CUR_VERT_POSN to zero.
            //
            cur_size = (unsigned long)(64 - PGL_cursor.height + PGL_cursor.hot_y - y);
            cur_pos = 0;
            cur_offset = PGL_cursor.cur_offset + (unsigned long)((PGL_cursor.hot_y - y) * 2);
        }
        else
        {
            //
            // Apply normal cursor defining parameters when y >= hot_y
            //
            cur_size = (unsigned long)(64 - PGL_cursor.height);
            cur_pos = (unsigned long)(y - PGL_cursor.hot_y);
            cur_offset = PGL_cursor.cur_offset;
        }

        // Y quantities are in the upper word for each cursor register
        cur_size = cur_size << 16;
        cur_pos = cur_pos << 16;

        if (x < PGL_cursor.hot_x)
        {
            //
            // Set the cursor position by adding (hot_x - x) cursor pixels to
            // CUR_HORZ_OFF. As in the Y dimension case, set CUR_HORZ_POSN to
            // zero.
            //
            cur_size = (unsigned long)(cur_size | (64 - PGL_cursor.width + PGL_cursor.hot_x - x));
        }
        else
        {
            //
            // Apply normal cursor defining parameters when x >= hot_x
            //
            cur_size = (unsigned long)(cur_size | (64 - PGL_cursor.width));
            cur_pos = (unsigned long)(cur_pos | (x - PGL_cursor.hot_x));
        }

        // write cursor positioning parameters to the cursor registers
        regw(CUR_OFFSET, cur_offset);
        regw(CUR_HORZ_VERT_OFF, cur_size);
        regw(CUR_HORZ_VERT_POSN, cur_pos);
    }
}

/* --------------------------------------------------------------------------
  PGL_gethwcursorpos - get the hardware cursor position

  The location is retrieved from static variables which are updated by
  calling PGL_sethwcurpos().
-------------------------------------------------------------------------- */
void PGL_gethwcursorpos(PGL_point *position)
{
    position->x = PGL_cursor.current_x;
    position->y = PGL_cursor.current_y;
}

