/****************************************************************************
 *                                                                          *
 *  VTGTDECTECT                                                             *
 *        This file has some VT/GT detect functions, including              *
 *        rev B detection.                                                  *
 *                                                                          *
 * Copyright (c) 1994-1998 ATI Technologies Inc. All rights reserved.       *
 ****************************************************************************/
#include "defines.h"
#include "atim64.h"
#include "atim64vt.h"


unsigned short int chip_id[20]= {
							0x00d7, // GX
							0x0057, // CX
							0x4354, // CT
							0x5654, // VT
							0x5656, // VT4
							0x4754, // 3D RAGE I, II
							0x4755, // 3D RAGE II+

							0x4756, // 3D RAGE IIC (PQFP, PCI 33 MHz)
							0x475A, // 3D RAGE IIC (PQFP, PCI 66 MHz)
							0x4757, // 3D RAGE IIC (BGA, PCI 66 MHz)
							0x4759, // 3D RAGE IIC (BGA, PCI 33 MHz)

							0x4742, // 3D RAGE PRO (BGA package, AGP1x2x)
							0x4744, // 3D RAGE PRO (BGA package, AGP1x only)
							0x4749, // 3D RAGE PRO (BGA package, PCI 33 MHz only)
							0x4750, // 3D RAGE PRO (PQFP package, PCI)
							0x4751, // 3D RAGE PRO (PQFP package, PCI, limited 3D)

							0x4C42, // 3D RAGE LT PRO
							0x4C44, // 3D RAGE LT PRO
							0x4C50  // 3D RAGE LT PRO

						};

#define MAX_CHIP_IDS 19
#define FIRST_3D_CHIP 5
#define FIRST_PRO 11


/****************************************************************************
 *                                                                          *
 * is_vtgt                                                                  *
 *  Function: checks if the installed chip is a VT/GT                       *
 *    Inputs: none                                                          *
 *   Outputs: returns 0 if a VT/GT is not detected                          *
 *            returns 1 if a VT/GT is detected                              *
 *                                                                          *
 ****************************************************************************/
unsigned int is_vtgt()
{
	unsigned int chipid;
	int slappy;
	
	chipid = ior32(CONFIG_CHIP_ID);

	for(slappy=3;slappy < MAX_CHIP_IDS; slappy++)
	{
		if((unsigned short)chipid == chip_id[slappy])
		{
			return(1);
		}
	}

	return(0);

} // end is_vtgt

/****************************************************************************
 *                                                                          *
 * is_gt                                                                    *
 *  Function: checks if the installed chip is a GT                          *
 *    Inputs: none                                                          *
 *   Outputs: returns 0 if a GT is not detected                             *
 *            returns 1 if a GT is detected                                 *
 *                                                                          *
 ****************************************************************************/
unsigned int is_gt()
{
	unsigned long chipid;
	int slappy;
	
	chipid = ior32(CONFIG_CHIP_ID);

	for(slappy=FIRST_3D_CHIP;slappy < MAX_CHIP_IDS; slappy++)
	{
		if((unsigned short)chipid == chip_id[slappy])
		{
			return(1);
		}
	}

	return(0);

} // end is_vtgt

/******************************************************************************
 *                                                                            *
 * is_gtb                                                                     *
 *  Function: checks if the installed chip is GTB                             *
 *    Inputs: none                                                            *
 *   Outputs: returns 0 if a GTB is not detected                              *
 *            returns 1 if a GTB is detected                                  *
 *                                                                            *
 ******************************************************************************/
unsigned int is_gtb()
{
	unsigned long chipid;
	int slappy;
	
	chipid = ior32(CONFIG_CHIP_ID);

	for(slappy=FIRST_3D_CHIP;slappy < FIRST_PRO; slappy++)
	{
		if((unsigned short)chipid == chip_id[slappy])
		{
			return(1);
		}
	}

	return(0);

} // end is_gtb

/******************************************************************************
 *                                                                            *
 * is_gtbplus                                                                 *
 *  Function: checks if the installed chip is RAGE II+/C                      *
 *    Inputs: none                                                            *
 *   Outputs: returns 0 if a RAGE II+/C is not detected                       *
 *            returns 1 if a RAGE II+/C is detected                           *
 *                                                                            *
 ******************************************************************************/
unsigned int is_gtbplus()
{
	unsigned long chipid;
	int slappy;
	
	chipid = ior32(CONFIG_CHIP_ID);
	chipid = chipid & 0x0000FFFF;
	
	for(slappy=6;slappy < FIRST_PRO; slappy++)
	{
		if((unsigned short)chipid == chip_id[slappy])
		{
			return(1);
		}
	}
	
	return(0);

} // end is_gtbplus


/******************************************************************************
 *                                                                            *
 * is_revb_or_greater                                                         *
 *  Function: checks the ASIC_ID to see whether or not the installed chip     *
 *            is a VT/GT rev B or greater.  This affects some of the          *
 *            multimedia registers (their layout/location).  This function    *
 *            will set the global variable ISREVB to true if a revB chip      *
 *            is detected.                                                    *
 *    Inputs: none                                                            *
 *   Outputs: returns 0 if a revB is not detected                             *
 *            returns 1 if a revB is detected                                 *
 *                                                                            *
 ******************************************************************************/
unsigned int is_revb_or_greater()
{
	unsigned int chipid;
	unsigned int test = 0;

	chipid = ior32(CFG_CHIP_REV);

	chipid = (chipid & 0x000000FF);
	
	switch((unsigned short)chipid)
	{

		case VTGTB1S1:
		case GTB1U1:
		case GTB1S2:
		case GTB2U1:
		case GTB2U2:
		case GTB2U3:
		case GTBC:
		case GTIIIC1U1:
		case GTIIIC1U2:
		case GTIIIC2U1:
		case GTIIIC2U2: 
		case GTIIIC2U3: 
		case LTPRO:     test = 1; // set to TRUE, as a revB or > is detected
						break;

		default: test = 0;
				 break;
	}

	return(test);

} // end is_revb_or_greater

/***************************************************************************    
 *                                                                         *
 *   Function: is_pro()                                                    *
 *    Purpose: Detects if the installed card has a 3D RAGE PRO ASIC        *
 *     Inputs: None                                                        *
 *    Outputs: TRUE (1) if 3D RAGE PRO is installed                        *
 *             FALSE (0) if 3D RAGE PRO not installed                      *
 *                                                                         *
 ***************************************************************************/
unsigned int is_pro()
{   
	int slappy, retval = 0;
	unsigned int chiprev = 0;

	chiprev = ior32(CONFIG_CHIP_ID);

	for(slappy = FIRST_PRO; slappy < MAX_CHIP_IDS; slappy++)
	{
		if(chip_id[slappy] == (unsigned short)chiprev)
		{
			retval = 1;
		}
	}

	return(retval);

} // end of is_pro()


/***************************************************************************    
 *                                                                         *
 *   Function: pro_overlay_init()                                          *
 *    Purpose: The 3D RAGE PRO requires special co-efficient registers     *
 *             to be initialized with particular values.  This function    *
 *             simply writes the appropriate values to the co-efficient    *
 *             registers.                                                  *
 *     Inputs: None                                                        *
 *    Outputs: None                                                        *
 *                                                                         *
 ***************************************************************************/
void pro_overlay_init() 
{

	wait_for_fifo (6);

	// set the scaler co-efficient registers
	regw (SCALER_COLOUR_CNTL, (0x00) | (0x10 << 8) | (0x10 << 16) );
	regw (SCALER_H_COEFF0, (0x00) | (0x20 << 8) );
	regw (SCALER_H_COEFF1, (0x0D) | (0x20 << 8) | (0x06 << 16) | (0x0D << 24) );
	regw (SCALER_H_COEFF2, (0x0D) | (0x1C << 8) | (0x0A << 16) | (0x0D << 24) );
	regw (SCALER_H_COEFF3, (0x0C) | (0x1A << 8) | (0x0E << 16) | (0x0C << 24) );
	regw (SCALER_H_COEFF4, (0x0C) | (0x14 << 8) | (0x14 << 16) | (0x0C << 24) );

	return;

} // end of pro_overlay_init


