/*==========================================================================
* ATTR.C - Copyright (c) 1993-94 ATI Technologies Inc. All rights reserved *
*                                                                          *
* PGL functions that control color and mix settings.                       *
* ======================================================================== */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <conio.h>

#include "..\inc\atim64.h"
#include "..\inc\pgl.h"
#include "..\inc\pglglob.h"

/* --------------------------------------------------------------------------
  PGL_setcolordepth - control depth component of internal PGL_modecfg
                      structure.

  This function can be used to directly control PGL_modecfg.depth variable
  used by PGL_getcolorcode() to return a real color value. This function
  is useful for the color weight where more than one choice exists. Valid
  values are shown below:

  bpp   color weight        depth value
  ----  ------------------  -----------
   4    palettized               0
   8    palettized               0
   15   5R-5G-5B                555
   16   5R-6G-5B                565
   24   8R-8G-8B (RGB)           0  (DEPTH_24_RGB)
   24   8B-8G-8R (BGR)           1  (DEPTH_24_BGR)
   32   8R-8G-8B-8a (RGBa)       0  (DEPTH_32_RGBA)
   32   8a-8R-8G-8B (aRGB)       1  (DEPTH_32_ARGB)
   32   8B-8G-8R-8a (BGRa)       2  (DEPTH_32_BGRA)
   32   8a-8B-8G-8R (aBGR)       3  (DEPTH_32_ABGR)

  Non-valid values will yield undefined results. The current 'depth' value
  can be retrieved by using PGL_getmodeconfig(). Note that 15 bpp is
  represented as 16 bpp with a color weight of 555 in the PGL.
-------------------------------------------------------------------------- */
void PGL_setcolordepth(int depth)
{
    PGL_modecfg.depth = depth;
}

/* --------------------------------------------------------------------------
  PGL_getcolorcode - return the color code for the given generic color.

  A "generic" color value is converted into a real color value for PGL
  functions PGL_setfgcolor() and PGL_getfgcolor() to make color setting
  video mode independant. Typically, the returned value is used as the
  input value to PGL_setfgcolor() or PGL_setbgcolor().
-------------------------------------------------------------------------- */
unsigned long PGL_getcolorcode(int generic_color)
{
    unsigned long color;

    // insure that the generic color is capped to the total
    generic_color = generic_color & 0x0f;

    // select appropriate generic color array based on color depth & weight
    switch(PGL_modecfg.bpp)
    {
        case 4:
        case 8:
            color = PGL_generic.colors_4_8[generic_color & 0x0f];
            break;

        case 16:
            switch(PGL_modecfg.depth)
            {
                case 555: // 16 bpp 555 weight
                    color = PGL_generic.colors_16_555[generic_color & 0x0f];
                    break;

                case 565: // 16 bpp 565 weight
                    color = PGL_generic.colors_16_565[generic_color & 0x0f];
                    break;
            }
            break;

        case 24:
            switch(PGL_modecfg.depth)
            {
                case DEPTH_24_RGB: // 24 bpp RGB weight
                    color = PGL_generic.colors_24_RGB[generic_color & 0x0f];
                    break;

                case DEPTH_24_BGR: // 24 bp BGR weight
                    color = PGL_generic.colors_24_BGR[generic_color & 0x0f];
                    break;
            }
            break;

        case 32:
            switch(PGL_modecfg.depth)
            {
                case DEPTH_32_RGBA: // 32 bpp RGBa weight
                    color = PGL_generic.colors_32_RGBA[generic_color & 0x0f];
                    break;

                case DEPTH_32_ARGB: // 32 bpp aRGB weight
                    color = PGL_generic.colors_32_ARGB[generic_color & 0x0f];
                    break;

                case DEPTH_32_BGRA: // 32 bpp BGRa weight
                    color = PGL_generic.colors_32_BGRA[generic_color & 0x0f];
                    break;

                case DEPTH_32_ABGR: // 32 bpp aBGR weight
                    color = PGL_generic.colors_32_ABGR[generic_color & 0x0f];
                    break;
            }
            break;
    }

    return (color);
}

/* --------------------------------------------------------------------------
  PGL_setfgcolor - set foreground color
-------------------------------------------------------------------------- */
void PGL_setfgcolor(unsigned long color)
{
    PGL_attr.fgcolor = color;
    PGL_waitforfifo(1);
    regw(DP_FRGD_CLR, color);
}

/* --------------------------------------------------------------------------
  PGL_getfgcolor - get foreground color
-------------------------------------------------------------------------- */
unsigned long PGL_getfgcolor(void)
{
    return (PGL_attr.fgcolor);
}

/* --------------------------------------------------------------------------
  PGL_setbgcolor - set background color
-------------------------------------------------------------------------- */
void PGL_setbgcolor(unsigned long color)
{
    PGL_attr.bgcolor = color;
    PGL_waitforfifo(1);
    regw(DP_BKGD_CLR, color);
}
                                                                
/* --------------------------------------------------------------------------
  PGL_getbgcolor - get background color
-------------------------------------------------------------------------- */
unsigned long PGL_getbgcolor(void)
{
    return (PGL_attr.bgcolor);
}

/* --------------------------------------------------------------------------
  PGL_setfgmix - set the current foreground mix
-------------------------------------------------------------------------- */
void PGL_setfgmix(int mix)
{
    unsigned long temp;

    PGL_waitforfifo(1);
    temp = (unsigned long)(mix);
    temp = temp << 16;
    temp = (unsigned long)(temp | PGL_attr.bgmix);
    regw(DP_MIX, temp);
    PGL_attr.fgmix = mix;
}

/* --------------------------------------------------------------------------
  PGL_getfgmix - get the current foreground mix
-------------------------------------------------------------------------- */
int PGL_getfgmix(void)
{
    return (PGL_attr.fgmix);
}

/* --------------------------------------------------------------------------
  PGL_setbgmix - set the current background mix
-------------------------------------------------------------------------- */
void PGL_setbgmix(int mix)
{
    unsigned long temp;

    PGL_waitforfifo(1);
    temp = (unsigned long)(PGL_attr.fgmix);
    temp = temp << 16;
    temp = (unsigned long)(temp | mix);
    regw(DP_MIX, temp);
    PGL_attr.bgmix = mix;
}

/* --------------------------------------------------------------------------
  PGL_getbgmix - set the current background mix

  See the description in PGL_setbgmix().
-------------------------------------------------------------------------- */
int PGL_getbgmix(void)
{
    return (PGL_attr.bgmix);
}

/* --------------------------------------------------------------------------
  PGL_setwritemask - set the current write mask
-------------------------------------------------------------------------- */
void PGL_setwritemask(unsigned long mask)
{
    PGL_waitforfifo(1);
    regw(DP_WRITE_MASK, mask);
    PGL_attr.writemask = mask;
}

/* --------------------------------------------------------------------------
  PGL_getwritemask - get the current write mask
-------------------------------------------------------------------------- */
unsigned long PGL_getwritemask(void)
{
    return (PGL_attr.writemask);
}

/* --------------------------------------------------------------------------
  PGL_getprimarycolor - return primary color component (R, G, B) from a given
                        color

  This function returns the primary color component from a given color. For
  4 and 8 bpp palettized modes, the given color value is used as an index
  to read the corresponding entry in the current palette table.
-------------------------------------------------------------------------- */
unsigned long PGL_getprimarycolor(PGL_primarycolor primarycolor,
                                  unsigned long color)
{
    unsigned long primary;
    PGL_palette entry;

	switch (PGL_modecfg.bpp)
	{
        case 4:
        case 8:
            entry = PGL_getpalette((int)(color & 0xff));
            switch (primarycolor)
            {
                case RED:   primary = (unsigned long)(entry.red);   break;
                case GREEN: primary = (unsigned long)(entry.green); break;
                case BLUE:  primary = (unsigned long)(entry.blue);  break;
            }
            primary = primary & 0xff;
            break;

        case 16:
            switch(PGL_modecfg.depth)
            {
                case 555: // 16 bpp with 555 weight (15 bpp)
                    switch (primarycolor)
                    {
                        case RED:   primary = (color >> 10) & 0x1f; break;
                        case GREEN: primary = (color >> 5) & 0x1f;  break;
                        case BLUE:  primary = color & 0x1f;         break;
                    }
                    break;

                case 565: // 565 weight
                    switch (primarycolor)
                    {
                        case RED:   primary = (color >> 11) & 0x1f; break;
                        case GREEN: primary = (color >> 5) & 0x3f;  break;
                        case BLUE:  primary = color & 0x1f;         break;
                    }
                    break;
            }
            break;

        case 24:
            switch(PGL_modecfg.depth)
            {
                case DEPTH_24_RGB: // RGB weight
                    switch (primarycolor)
                    {
                        case RED:   primary = (color >> 16) & 0xff; break;
                        case GREEN: primary = (color >> 8) & 0xff;  break;
                        case BLUE:  primary = color & 0xff;         break;
                    }
                    break;

                case DEPTH_24_BGR: // BGR weight
                    switch (primarycolor)
                    {
                        case RED:   primary = color & 0xff;         break;
                        case GREEN: primary = (color >> 8) & 0xff;  break;
                        case BLUE:  primary = (color >> 16) & 0xff; break;
                    }
                    break;
            }
            break;

        case 32:
            switch(PGL_modecfg.depth)
            {
                case DEPTH_32_RGBA: // RGBa weight
                    switch (primarycolor)
                    {
                        case RED:   primary = (color >> 24) & 0xff; break;
                        case GREEN: primary = (color >> 16) & 0xff; break;
                        case BLUE:  primary = (color >> 8) & 0xff;  break;
                    }
                    break;

                case DEPTH_32_ARGB: // aRGB weight
                    switch (primarycolor)
                    {
                        case RED:   primary = (color >> 16) & 0xff; break;
                        case GREEN: primary = (color >> 8) & 0xff;  break;
                        case BLUE:  primary = color & 0xff;         break;
                    }
                    break;

                case DEPTH_32_BGRA: // BGRa weight
                    switch (primarycolor)
                    {
                        case RED:   primary = (color >> 8) & 0xff;  break;
                        case GREEN: primary = (color >> 16) & 0xff; break;
                        case BLUE:  primary = (color >> 24) & 0xff; break;
                    }
                    break;

                case DEPTH_32_ABGR: // aBGR weight
                    switch (primarycolor)
                    {
                        case RED:   primary = color & 0xff;         break;
                        case GREEN: primary = (color >> 8) & 0xff;  break;
                        case BLUE:  primary = (color >> 16) & 0xff; break;
                    }
                    break;
            }
            break;
    }

    return (primary);
}

/* --------------------------------------------------------------------------
  PGL_makecolor - return color from separate color components (R, G, B)

  This function takes the given color components (RED, GREEN, and BLUE) and
  combines them into a single color value (15, 16, 24, and 32 bpp modes).
  For palettized modes (4 and 8 bpp), the color components will be compared
  with the current palette entries and the matching index will be returned.
  If no match is found, NOCOLORMATCH (0xffff) is returned.
-------------------------------------------------------------------------- */
unsigned long PGL_makecolor(unsigned long r, unsigned long g, unsigned long b)
{
    unsigned long color;
    PGL_palette entry;
    int index, maxentry;

	switch (PGL_modecfg.bpp)
	{
        case 4:
        case 8:
            color = NOCOLORMATCH;
            maxentry = 256;
            if (PGL_modecfg.bpp == 4)
            {
                maxentry = 16;
            }
            for (index = 0; index < maxentry; index++)
            {
                entry = PGL_getpalette(index);
                if (entry.red == (int)(r))
                {
                    if (entry.green == (int)(g))
                    {
                        if (entry.blue == (int)(b))
                        {
                            color = (unsigned long)(index);
                            index = 256;
                        }
                    }
                }
            }
            break;

        case 16:
            switch(PGL_modecfg.depth)
            {
                case 555: // 555 weight
                    color = (r << 10) | (g << 5) | b;
                    break;

                case 565: // 565 weight
                    color = (r << 11) | (g << 5) | b;
                    break;
            }
            break;

        case 24:
            switch(PGL_modecfg.depth)
            {
                case DEPTH_24_RGB: // RGB weight
                    color = (r << 16) | (g << 8) | b;
                    break;

                case DEPTH_24_BGR: // BGR weight
                    color = (b << 16) | (g << 8) | r;
                    break;
            }
            break;

        case 32:
            switch(PGL_modecfg.depth)
            {
                case DEPTH_32_RGBA: // RGBa weight
                    color = (r << 24) | (g << 16) | (b << 8);
                    break;

                case DEPTH_32_ARGB: // aRGB weight
                    color = (r << 16) | (g << 8) | b;
                    break;

                case DEPTH_32_BGRA: // BGRa weight
                    color = (r << 8) | (g << 16) | (b << 24);
                    break;

                case DEPTH_32_ABGR: // aBGR weight
                    color = r | (g << 8) | (b << 16);
                    break;
            }
            break;
    }

    return (color);
}

