; ===========================================================================
; IOREG.ASM
;
; Mach 64 Interface routines to perform IO access with different IO base
; addresses. Input parameters are fetched off the stack.
;
; Compiling:
;   masm /Ml /D<memory model> ioreg.asm;
;       <memory model> = mem_S for SMALL model,
;                        mem_M for MEDIUM model,
;                        mem_L for LARGE model
;
; Copyright (c) 1994-1995 ATI Technologies Inc. All rights reserved
; ===========================================================================

include atim64.inc

CHKIO_2ec   equ     0ECh
CHKIO_1c8   equ     0C8h
CHKIO_1cc   equ     0CCh
CHKIO_MASK  equ     0FFFCh

IFDEF mem_S
PARM        equ     4   ; passed parameters start at bp+4 for small model
ELSE
PARM        equ     6   ; passed parameters start at bp+6 for other models
ENDIF

IFDEF mem_S
.MODEL  SMALL, C
ELSEIFDEF mem_M
.MODEL  MEDIUM, C
ELSE
.MODEL  LARGE, C
ENDIF

.DATA

IOBase      DW      02ECh
IOType      DW      0

.CODE
.286

; Macro for 'call' model handling
Mcall       macro   routine
IFDEF mem_S
            call    NEAR PTR routine
ELSE
            call    FAR PTR routine
ENDIF
            endm

; ---------------------------------------------------------------------------
; INIT_IO_REG
;
; Initialize I/O base and type for I/O register functions. Input parameters
; are fetched from the stack.
;
; Inputs : WORD IO base address
;          WORD IO type (0 = standard, 1 = floating)
;
; Outputs: none
;
; This function MUST be called before any of the IO routines can be used.
; ---------------------------------------------------------------------------
            public  init_io_reg

IFDEF mem_S
init_io_reg proc    near
ELSE
init_io_reg proc    far
ENDIF
            ; create frame pointer
            push    bp
            mov     bp, sp

            ; initialize base IO address from input parameter on stack
            push    ax
            mov     ax, WORD PTR [bp+PARM]
            mov     IOBase, ax
            mov     ax, WORD PTR [bp+PARM+2]
            and     ax, 1
            mov     IOType, ax
            pop     ax

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

init_io_reg endp

; ---------------------------------------------------------------------------
; IO_STANDARD_TO_FLOAT - convert standard I/O to floating I/O (byte offset)
;
; With floating I/O enabled, the memory mapped register (MMR) scheme is used
; instead of the normal I/O decoding scheme. This routine accepts an I/O
; decoding scheme byte offset and will convert it to floating MMR scheme.
; This will allow the IOW?() and IOR?() functions to operate transparently to
; the application. Note that for this to work, the values in ATIM64.H and
; ATIM64.INC MUST BE IDENTICAL. Also, the I/O and MMR values must be byte
; offsets.
;
; Inputs : AX = standard I/O byte offset using standard I/O scheme
;
; Outputs: AX = converted I/O byte offset using MMR scheme
; ---------------------------------------------------------------------------
            public io_standard_to_float

IFDEF mem_S
io_standard_to_float proc near
ELSE
io_standard_to_float proc far
ENDIF
            ; save used registers
            push    bx
            push    dx

            ; save original input value
            mov     dx, ax

            ; strip lower two bits (in case 0, 1, 2, or 3 was added to offset)
            and     ax, 0FFFCh

            ; Group 1
            cmp     ax, ioCRTC_H_TOTAL_DISP_ALIAS
            je      iogrp1
            mov     bx, ioCRTC_GEN_CNTL
            sub     bx, ax
            jc      niogrp1
iogrp1:
            mov     bx, 0                       ; delta MMR = 0h
            jmp     io_convert

niogrp1:
            ; Group 2
            mov     bx, ioOVR_WID_TOP_BOTTOM
            sub     bx, ax
            jc      @F
            mov     bx, 8h                      ; delta MMR = 8h
            jmp     io_convert

@@:
            ; Group 3
            mov     bx, ioCUR_HORZ_VERT_OFF
            sub     bx, ax
            jc      @F
            mov     bx, 0Dh                     ; delta MMR = 0Dh
            jmp     io_convert

@@:
            ; Group 4
            mov     bx, ioSCRATCH_REG1
            sub     bx, ax
            jc      @F
            mov     bx, 10h                     ; delta MMR = 10h
            jmp     io_convert

@@:
            ; Group 5
            cmp     ax, ioCLOCK_CNTL
            jne     @F
            mov     bx, 12h                     ; delta MMR = 12h
            jmp     io_convert

@@:
            ; Group 6
            cmp     ax, ioBUS_CNTL
            jne     @F
            mov     bx, 15h                     ; delta MMR = 15h
            jmp     io_convert

@@:
            ; Group 7
            mov     bx, ioMEM_VGA_RP_SEL
            sub     bx, ax
            jc      @F
            mov     bx, 18h                     ; delta MMR = 18h
            jmp     io_convert

@@:
            ; Group 8
            mov     bx, ioDAC_CNTL
            sub     bx, ax
            jc      @F
            mov     bx, 19h                     ; delta MMR = 19h
            jmp     io_convert

@@:
            ; Group 9
            cmp     ax, ioGEN_TEST_CNTL
            jne     @F
            mov     bx, 1Bh                     ; delta MMR = 1Bh
            jmp     io_convert

@@:
            ; Group 10
            mov     bx, ioCONFIG_STAT1          ; same as ioCRC_SIG
            sub     bx, ax
            jc      io_nomatch
            mov     bx, 1Dh                     ; delta MMR = 1Dh

io_convert:
            ; new_value = (((old_value >> 10) + dMMR) << 2) + register offset
            shr     ax, 10
            add     ax, bx
            shl     ax, 2
            and     dx, 3                       ; add any register offsets
            add     ax, dx                      ;  0, 1, 2, or 3
            jmp     io_ret

io_nomatch:
            ; restore original value in ax
            mov     ax, dx

io_ret:
            ; restore used registers
            pop     dx
            pop     bx

            ret

io_standard_to_float endp

; ---------------------------------------------------------------------------
; CONVERT_IO_ADDR - convert given physical I/O address to byte offset
;
; XXEC + 0,1,2,3 = 2ECh based address
; XXC8 + 0,1,2,3 = 1C8h based address
; XXCC + 0,1,2,3 = 1CCh based address
;
; Inputs : AX = I/O address to convert
;
; Outputs: AX = I/O byte offset
; ---------------------------------------------------------------------------
            public convert_io_addr

IFDEF mem_S
convert_io_addr proc near
ELSE
convert_io_addr proc far
ENDIF
            ; save used registers
            push    bx

            ; place I/O address in bx
            mov     bx, ax

            ; remove any register offsets
            and     bx, CHKIO_MASK

            ; check if address is 2ech based
            cmp     bl, CHKIO_2ec
            jne     @F
            sub     ax, 2ECh                    ; convert to byte offset
            jmp     cia_ret

@@:
            ; check if address is 1c8h based
            cmp     bl, CHKIO_1c8
            jne     @F
            sub     ax, 1C8h                    ; convert to byte offset
            jmp     cia_ret

@@:
            ; check if address is 1cch based
            cmp     bl, CHKIO_1cc
            jne     cia_ret
            sub     ax, 1CCh                    ; convert to byte offset

cia_ret:
            ; restore used registers
            pop     bx

            ret

convert_io_addr endp

; ---------------------------------------------------------------------------
; IOW8
;
; Write 8 bits to one of the four bytes of a 32 bit I/O mapped register.
; This is the 8 bit version of IOW32().
;
; Inputs : WORD IO address (stack)
;          WORD 8 bit data (stack)
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  iow8

IFDEF mem_S
iow8        proc    near
ELSE
iow8        proc    far
ENDIF
            ; create frame pointer
            push    bp
            mov     bp, sp

            ; save used registers
            push    ax
            push    bx
            push    dx

            ; get input parameters
            mov     dx, WORD PTR [bp+PARM]      ; get I/O address/index
            mov     bx, WORD PTR [bp+PARM+2]    ; get 8 bit data to write
            xor     bh, bh                      ; mask data to 8 bit size

            ; place I/O address in ax
            mov     ax, dx

            ; check if given I/O address is address or byte offset
            and     dx, CHKIO_MASK              ; remove any register offsets
            cmp     dl, 0                       ; if 0 -> byte offset
            je      @F
            Mcall   convert_io_addr             ; convert to byte offset

@@:
            ; convert standard I/O (for index only) if I/O type is floating
            cmp     IOType, 0                   ; if 0 -> standard I/O
            je      @F
            Mcall   io_standard_to_float

@@:
            ; write 8 bits to I/O port
            add     ax, IOBase                  ; address is index + base
            mov     dx, ax                      ; get I/O address
            mov     al, bl                      ; get 8 bit data to write
            out     dx, al                      ; output

            ; restore used registers
            pop     dx
            pop     bx
            pop     ax

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

iow8        endp

; ---------------------------------------------------------------------------
; IOR8
;
; Read 8 bits from one of the four bytes of a 32 bit I/O mapped register.
; This is the 8 bit version of IOR32().
;
; Inputs : WORD IO address (stack)
;
; Outputs: AX = 8 bit data
; ---------------------------------------------------------------------------
            public  ior8

IFDEF mem_S
ior8        proc    near
ELSE
ior8        proc    far
ENDIF
            ; create frame pointer
            push    bp
            mov     bp, sp

            ; save used registers
            push    dx

            ; get input parameters
            mov     dx, WORD PTR [bp+PARM]      ; get I/O address/index

            ; place I/O address in ax
            mov     ax, dx

            ; check if given I/O address is address or byte offset
            and     dx, CHKIO_MASK              ; remove any register offsets
            cmp     dl, 0                       ; if 0 -> byte offset
            je      @F
            Mcall   convert_io_addr             ; convert to byte offset

@@:
            ; convert standard I/O (for index only) if I/O type is floating
            cmp     IOType, 0                   ; if 0 -> standard I/O
            je      @F
            Mcall   io_standard_to_float

@@:
            ; read 8 bits from I/O port
            add     ax, IOBase                  ; address is index + base
            mov     dx, ax                      ; get I/O address
            in      al, dx                      ; input
            xor     ah, ah                      ; insure 8 bit data size

            ; restore used registers
            pop     dx

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

ior8        endp

; ---------------------------------------------------------------------------
; IOW16
;
; Write 16 bits to one of the two words of a 32 bit I/O mapped register.
; This is the 16 bit version of IOW32().
;
; Inputs : WORD IO address (stack)
;          WORD 16 bit data (stack)
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  iow16

IFDEF mem_S
iow16       proc    near
ELSE
iow16       proc    far
ENDIF
            ; create frame pointer
            push    bp
            mov     bp, sp

            ; save used registers
            push    ax
            push    bx
            push    dx

            ; get input parameters
            mov     dx, WORD PTR [bp+PARM]      ; get I/O address/index
            mov     bx, WORD PTR [bp+PARM+2]    ; get 16 bit data to write

            ; place I/O address in ax
            mov     ax, dx

            ; check if given I/O address is address or byte offset
            and     dx, CHKIO_MASK              ; remove any register offsets
            cmp     dl, 0                       ; if 0 -> byte offset
            je      @F
            Mcall   convert_io_addr             ; convert to byte offset

@@:
            ; convert standard I/O (for index only) if I/O type is floating
            cmp     IOType, 0                   ; if 0 -> standard I/O
            je      @F
            Mcall   io_standard_to_float

@@:
            ; write 16 bits to I/O port
            add     ax, IOBase                  ; address is index + base
            mov     dx, ax                      ; get I/O address
            mov     ax, bx                      ; get 16 bit data to write
            out     dx, ax                      ; output

            ; restore used registers
            pop     dx
            pop     bx
            pop     ax

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

iow16       endp

; ---------------------------------------------------------------------------
; IOR16
;
; Read 16 bits from one of the two words of a 32 bit I/O mapped register.
; This is the 16 bit version of IOR32().
;
; Inputs : WORD IO address (stack)
;
; Outputs: AX = 16 bit data
; ---------------------------------------------------------------------------
            public  ior16

IFDEF mem_S
ior16       proc    near
ELSE
ior16       proc    far
ENDIF
            ; create frame pointer
            push    bp
            mov     bp, sp

            ; save used registers
            push    dx

            ; get input parameters
            mov     dx, WORD PTR [bp+PARM]      ; get I/O address/index

            ; place I/O address in ax
            mov     ax, dx

            ; check if given I/O address is address or byte offset
            and     dx, CHKIO_MASK              ; remove any register offsets
            cmp     dl, 0                       ; if 0 -> byte offset
            je      @F
            Mcall   convert_io_addr             ; convert to byte offset

@@:
            ; convert standard I/O (for index only) if I/O type is floating
            cmp     IOType, 0                   ; if 0 -> standard I/O
            je      @F
            Mcall   io_standard_to_float

@@:
            ; read 16 bits from I/O port
            add     ax, IOBase                  ; address is index + base
            mov     dx, ax                      ; get I/O address
            in      ax, dx                      ; input

            ; restore used registers
            pop     dx

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

ior16       endp

; ---------------------------------------------------------------------------
; IOW32 - write to an I/O mapped register in 32 bit wide data.
;
; This function will write to an I/O register in 32 bits. This is done in
; two 16 bit writes - first to the low word io address, then to the upper
; word address. The function accepts the following input types:
;
; 1) Byte offsets from register nmenomics found in ATIM64.H/ATIM64.INC
; 2) 2ECh based register addresses from Mach64 GX I/O register specification
; 3) 1C8h based register addresses
; 4) 1CCh based register addresses
;
; XX00 + 0,1,2,3 = byte offset from ATIM64.H/ATIM64.INC
; XXEC + 0,1,2,3 = 2ECh based address
; XXC8 + 0,1,2,3 = 1C8h based address
;
; Note that Mach64 CX I/O register specification values (high 6 bit indices)
; are not accepted since it is not possible to access individual bytes within
; a dword using this system.
;
; Inputs : WORD  IO address (stack)
;          DWORD 32 bit data (stack)
;
; Outputs: none
; ---------------------------------------------------------------------------
            public  iow32

IFDEF mem_S
iow32       proc    near
ELSE
iow32       proc    far
ENDIF
            ; create frame pointer
            push    bp
            mov     bp, sp

            ; save used registers
            push    ax
            push    bx
            push    cx
            push    dx

            ; get input parameters
            mov     dx, WORD PTR [bp+PARM]      ; get I/O address/index
            mov     bx, WORD PTR [bp+PARM+2]    ; get low 16 bits of data
            mov     cx, WORD PTR [bp+PARM+4]    ; get high 16 bits of data

            ; place I/O address in ax
            mov     ax, dx

            ; check if given I/O address is address or byte offset
            and     dx, CHKIO_MASK              ; remove any register offsets
            cmp     dl, 0                       ; if 0 -> byte offset
            je      @F
            Mcall   convert_io_addr             ; convert to byte offset

@@:
            ; convert standard I/O (for index only) if I/O type is floating
            cmp     IOType, 0                   ; if 0 -> standard I/O
            je      @F
            Mcall   io_standard_to_float

@@:
            ; write 32 bits to I/O port
            add     ax, IOBase                  ; address is index + base
            mov     dx, ax                      ; get I/O address
            mov     ax, bx                      ; get low 16 bits of data
            out     dx, ax                      ; output
            inc     dx                          ; next word
            inc     dx
            mov     ax, cx                      ; get high 16 bits of data
            out     dx, ax                      ; output

            ; restore used registers
            pop     dx
            pop     cx
            pop     bx
            pop     ax

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

iow32       endp

; ---------------------------------------------------------------------------
; IOR32 - read from an I/O mapped register in 32 bit wide data.
;
; This function will read to an I/O register in 32 bits. This is done in
; two 16 bit reads - first to the low word io address, then to the upper
; word address. The function accepts the following input bases:
;
; 1) Byte offsets from register nmenomics found in ATIM64.H/ATIM64.INC
; 2) 2ECh based register addresses from Mach64 GX I/O register specification
; 3) 1C8h based register addresses
; 4) 1CCh based register addresses
;
; XX00 + 0,1,2,3 = byte offset from ATIM64.H/ATIM64.INC
; XXEC + 0,1,2,3 = 2ECh based address
; XXC8 + 0,1,2,3 = 1C8h based address
;
; Note that Mach64 CX I/O register specification values (high 6 bit indices)
; are not accepted since it is not possible to access individual bytes within
; a dword using this system.
;
; Inputs : WORD IO address (stack)
;
; Outputs: AX = low 16 bits of data
;          DX = high 16 bits of data
; ---------------------------------------------------------------------------
            public  ior32

IFDEF mem_S
ior32       proc    near
ELSE
ior32       proc    far
ENDIF
            ; create frame pointer
            push    bp
            mov     bp, sp

            ; save used registers
            push    bx

            ; get input parameters
            mov     dx, WORD PTR [bp+PARM]      ; get I/O address/index

            ; place I/O address in ax
            mov     ax, dx

            ; check if given I/O address is address or byte offset
            and     dx, CHKIO_MASK              ; remove any register offsets
            cmp     dl, 0                       ; if 0 -> byte offset
            je      @F
            Mcall   convert_io_addr             ; convert to byte offset

@@:
            ; convert standard I/O (for index only) if I/O type is floating
            cmp     IOType, 0                   ; if 0 -> standard I/O
            je      @F
            Mcall   io_standard_to_float

@@:
            ; read 32 bits from I/O port
            add     ax, IOBase                  ; address is index + base
            mov     dx, ax                      ; get I/O address
            in      ax, dx                      ; input low 16 bits of data
            mov     bx, ax                      ; save in bx
            inc     dx                          ; next word
            inc     dx
            in      ax, dx                      ; input high 16 bits of data
            mov     dx, ax                      ; place data in DX:AX
            mov     ax, bx

            ; restore used registers
            pop     bx

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

ior32       endp

            end

