/*==========================================================================
* LINEPAT.C - Copyright (c) 1993 ATI Technologies Inc. All rights reserved *
*                                                                          *
* PGL functions to set the line pattern.                                   *
* ======================================================================== */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <conio.h>

#include "..\inc\atim64.h"
#include "..\inc\pgl.h"
#include "..\inc\pglglob.h"

/* --------------------------------------------------------------------------
  PGL_setlinepattern - set the current line pattern by user definition

  Calling this function will replace the line pattern type selected in
  PGL_setlinetype(). The line pattern can be upto 32 bits long. The
  pattern is interpreted from LSB to MSB (to length bits) of 'linepattern'.
  The 'length' parameter must be 1 to 32. The pattern will remain
  unchanged if the 'length' parameter is invalid.

  PGL_setlinetype() must be called to enable user defined line patterns.
-------------------------------------------------------------------------- */
void PGL_setlinepattern(unsigned long linepattern, int length)
{
    if ((length > 32) || (length < 1)) return;

    /* Store given and converted patterns */
    if (PGL_modecfg.bpp == 4)
    {
        PGL_attr.linepattern = pgl_convertpattern(linepattern);
    }
    else
    {
        PGL_attr.linepattern = linepattern;
    }
    PGL_attr.linelength = length;
}

/* --------------------------------------------------------------------------
  PGL_setlinetype - set the current line pattern

  The line pattern used is determined by the line pattern type given or a
  by a user defined line pattern of upto 32 bits long:

  0 - User defined pattern set by PGL_setlinepattern()
  1 - Dotted line -- 1 on, 2 off
  2 - Short dashed line -- 5 on, 3 off
  3 - Dash-dot line -- 6 on, 4 off, 2 on, 4 off
  4 - Double dotted line -- 2 on, 4 off, 2 on, 8 off
  5 - Long dash line -- 9 on, 3 off
  6 - Dash double-dot line -- 8 on, 4 off, 2 on, 2 off, 2 on, 2 off
  7 - Solid line
  8 - Invisible line

  The line pattern will remain unchanged if the given parameter is invalid.
-------------------------------------------------------------------------- */
void PGL_setlinetype(int type)
{
    if ((type < 0) || (type > 8)) return;

    // no support for patterned lines in 24 bpp
    if (PGL_modecfg.bpp == 24)
    {
        // solid lines only
        type = 7;
    }

    switch(type)
    {
        case 1: PGL_setlinepattern(0x00000001, 3); break;
        case 2: PGL_setlinepattern(0x0000001f, 8); break;
        case 3: PGL_setlinepattern(0x00000c3f, 16); break;
        case 4: PGL_setlinepattern(0x000000c3, 16); break;
        case 5: PGL_setlinepattern(0x000001ff, 12); break;
        case 6: PGL_setlinepattern(0x000330ff, 20); break;
        case 7: PGL_setlinepattern(0x0000ffff, 16); break;
        case 8: PGL_setlinepattern(0x00000, 16); break;
    }
    PGL_attr.linetype = type;
}

